using System;
using System.Runtime.ConstrainedExecution;
using System.Reflection;
using System.Configuration;
using System.ServiceProcess;
using System.Threading;
using System.IO;
using System.Text;
using System.Diagnostics;
using System.Collections;
using System.Runtime.Remoting;
using Microsoft.Win32;
using System.Messaging;
using System.Collections.Generic;

namespace CSharpRecipes
{
	public class Toolbox
    {
        #region 20.1 Obsuga procesw zamknicia systemu, zarzdzania moc lub zmian w sesji uytkownika
        public static void PreventBadShutdown()
        {
            RegisterForSystemEvents();

            // zmiany trybu zarzdzania moc
            // zmiany sesji

            UnregisterFromSystemEvents();
        }

        public static void RegisterForSystemEvents()
        {
            // kadorazowe otrzymywanie ostatniego powiadomienia, gdy wtek zdarzenia
            // zostaje zamknity, by mc si wyrejestrowa
            SystemEvents.EventsThreadShutdown += 
                new EventHandler(OnEventsThreadShutdown);
            SystemEvents.PowerModeChanged +=
                new PowerModeChangedEventHandler(OnPowerModeChanged);
            SystemEvents.SessionSwitch +=
                new SessionSwitchEventHandler(OnSessionSwitch);
            SystemEvents.SessionEnding +=
                new SessionEndingEventHandler(OnSessionEnding);
            SystemEvents.SessionEnded +=
                new SessionEndedEventHandler(OnSessionEnded);
        }

        private static void UnregisterFromSystemEvents()
        {
            SystemEvents.EventsThreadShutdown -= 
                new EventHandler(OnEventsThreadShutdown);
            SystemEvents.PowerModeChanged -=
                new PowerModeChangedEventHandler(OnPowerModeChanged);
            SystemEvents.SessionSwitch -=
                new SessionSwitchEventHandler(OnSessionSwitch);
            SystemEvents.SessionEnding -=
                new SessionEndingEventHandler(OnSessionEnding);
            SystemEvents.SessionEnded -=
                new SessionEndedEventHandler(OnSessionEnded);
        }

        private static void OnEventsThreadShutdown(object sender, EventArgs e)
        {
            Debug.WriteLine("Wyczenie wtku zdarze systemowych, koniec powiadamiania.");

            // wyrejestrowanie wszystkich zdarze, poniewa wtek powiadamiania zostaje wyczony
            UnregisterFromSystemEvents();
        }

        private static void OnPowerModeChanged(object sender, PowerModeChangedEventArgs e)
        {
            // zmienia si tryb poboru mocy
            switch (e.Mode)
            {
                case PowerModes.Resume:
                    Debug.WriteLine("Tryb poboru mocy: system operacyjny wraca z trybu zawieszenia");
                    break;
                case PowerModes.StatusChange:
                    Debug.WriteLine("Tryb poboru mocy: Nastpia zmiana zwizana z poborem " +
                        "mocy (saba bateria, odczenie od prdu itp..)");
                    break;
                case PowerModes.Suspend:
                    Debug.WriteLine("Tryb poboru mocy: system operacyjny przechodzi w stan zawieszenia pracy");
                    break;
            }
        }

        private static void OnSessionSwitch(object sender, SessionSwitchEventArgs e)
        {
            // sprawdzenie przyczyny
            switch (e.Reason)
            {
                case SessionSwitchReason.ConsoleConnect:
                    Debug.WriteLine("Sesja podczona z konsoli");
                    break;
                case SessionSwitchReason.ConsoleDisconnect:
                    Debug.WriteLine("Sesja odczona z konsoli");
                    break;
                case SessionSwitchReason.RemoteConnect:
                    Debug.WriteLine("Podczona zdalna sesja");
                    break;
                case SessionSwitchReason.RemoteDisconnect:
                    Debug.WriteLine("Rozczona zdalna sesja");
                    break;
                case SessionSwitchReason.SessionLock:
                    Debug.WriteLine("Sesja zostaa zablokowana");
                    break;
                case SessionSwitchReason.SessionLogoff:
                    Debug.WriteLine("Uytkownik wylogowa si z sesji");
                    break;
                case SessionSwitchReason.SessionLogon:
                    Debug.WriteLine("Uytkownik zalogowa si do sesji");
                    break;
                case SessionSwitchReason.SessionRemoteControl:
                    Debug.WriteLine("Sesja przesza w lub ze stanu zdalnego");
                    break;
                case SessionSwitchReason.SessionUnlock:
                    Debug.WriteLine("Sesja zostaa odblokowana");
                    break;
            }
        }

        private static void OnSessionEnding(object sender, SessionEndingEventArgs e)
        {
            // true, aby anulowa zakocznie sesji na danie uytkownika; false w przeciwnym razie
            e.Cancel = false;
            // sprawdzenie przyczyny
            switch(e.Reason)
            {
                case SessionEndReasons.Logoff:
                    Debug.WriteLine("Koniec sesji z powodu wylogowania uytkownika");
                    break;
                case SessionEndReasons.SystemShutdown:
                    Debug.WriteLine("Koniec sesji z powodu zakoczenia pracy przez system operacyjny");
                    break;
            }
        }

        private static void OnSessionEnded(object sender, SessionEndedEventArgs e)
        {
            switch (e.Reason)
            {
                case SessionEndReasons.Logoff:
                    Debug.WriteLine("Sesja zakoczya si z powodu wylogowania uytkownika");
                    break;
                case SessionEndReasons.SystemShutdown:
                    Debug.WriteLine("Sesja zakoczya si z powodu zamknicia systemu operacyjnego");
                    break;
            }
        }

        #endregion

        #region 20.2 Sterowanie usug
        public static void TestServiceManipulation()
        {
            ServiceController scStateService = new ServiceController("System zdarze COM+");
            foreach (ServiceController sc in scStateService.DependentServices)
            {
                Console.WriteLine(scStateService.DisplayName + " zaley od: " + sc.DisplayName);
            }
            Console.WriteLine("Rodzaj usugi: " + scStateService.ServiceType.ToString());
            Console.WriteLine("Nazwa usugi: " + scStateService.ServiceName);
            Console.WriteLine("Nazwa wywietlana: " + scStateService.DisplayName);
            foreach (ServiceController sc in scStateService.ServicesDependedOn)
            {
                Console.WriteLine(scStateService.DisplayName + " zaley od: " + sc.DisplayName);
            }
            Console.WriteLine("Status: " + scStateService.Status);
            // zapisanie pocztkowego stanu
            ServiceControllerStatus originalState = scStateService.Status;
            TimeSpan serviceTimeout = TimeSpan.FromSeconds(60);
            // uruchomienie usugi, jeli jest zatrzymana
            if (scStateService.Status == ServiceControllerStatus.Stopped)
            {
                scStateService.Start();
                // odczekanie maksymalnie 60 sekund na uruchomienie usugi
                scStateService.WaitForStatus(ServiceControllerStatus.Running, serviceTimeout);
            }
            Console.WriteLine("Status: " + scStateService.Status);
            // jeli usuga jest wstrzymana - uruchomienie jej
            if (scStateService.Status == ServiceControllerStatus.Paused)
            {
                if(scStateService.CanPauseAndContinue)
                {
                    scStateService.Continue();
                    // odczekanie maksymalnie 60 sekund na uruchomienie usugi
                    scStateService.WaitForStatus(ServiceControllerStatus.Running, serviceTimeout);
                }
            }
            Console.WriteLine("Stan: " + scStateService.Status);

            // w tym momencie usuga powinna ju dziaa

            // czy mona j zatrzyma?
            if (scStateService.CanStop)
            {
                scStateService.Stop();
                // odczekanie maksymalnie 60 sekund na zatrzymanie usugi
                scStateService.WaitForStatus(ServiceControllerStatus.Stopped, serviceTimeout);
            }
            Console.WriteLine("Stan: " + scStateService.Status);

            // przywrcenie stanu pocztkowego
            switch (originalState)
            {
                case ServiceControllerStatus.Stopped:
                    if (scStateService.CanStop)
                    {
                        scStateService.Stop();
                    }
                    break;
                case ServiceControllerStatus.Running:
                    scStateService.Start();
                    // odczekanie maksymalnie 60 sekund na uruchomienie usugi
                    scStateService.WaitForStatus(ServiceControllerStatus.Running, serviceTimeout);
                    break;
                case ServiceControllerStatus.Paused:
                    // jeli usuga bya wstrzymana i jest zatrzymana, trzeba j zrestartowa, by mc j wstrzyma
                    if (scStateService.Status == ServiceControllerStatus.Stopped)
                    {
                        scStateService.Start();
                        // odczekanie maksymalnie 60 sekund na uruchomienie usugi
                        scStateService.WaitForStatus(ServiceControllerStatus.Running, serviceTimeout);
                    }
                    // wstrzymanie usugi
                    if (scStateService.CanPauseAndContinue)
                    {
                        scStateService.Pause();
                        // odczekanie maksymalnie 60 sekund na wstrzyamnie usugi
                        scStateService.WaitForStatus(ServiceControllerStatus.Paused, serviceTimeout);
                    }
                    break;
            }
            scStateService.Refresh();
            Console.WriteLine("Stan: " + scStateService.Status.ToString());

            // zamknicie
            scStateService.Close();
        }
        #endregion

        #region 20.3 Uzyskiwanie listy procesw, w ktrych zaadowano podzesp
        public static List<Process> GetProcessesAssemblyIsLoadedIn(string assemblyFileName)
        {
            List<Process> processList = new List<Process>();
            Process[] processes = Process.GetProcesses();
            foreach (Process p in processes)
            {
                foreach (ProcessModule module in p.Modules)
                {
                    if (module.ModuleName.Equals(assemblyFileName, StringComparison.OrdinalIgnoreCase))
                    {
                        processList.Add(p);
                        break;
                    }
                }
            }
            return processList;
        }
        #endregion

        #region 20.4 Uywanie kolejek komunikatw na lokalnej stacji roboczej
        public static void TestMessageQueue()
        {
            // UWAGA: Aby metoda dziaaa, trzeba skonfigurowa usugi Message Queue
            // Mona je doda w Dodaj/Usu komponenty systemu Windows
            try
            {
                // oto prawidowa skadnia dla kolejek stacji roboczej
                // MQWorker mqw = new MQWorker(@".\private$\MQWorkerQ");
                MQWorker mqw = new MQWorker(@".\MQWorkerQ");
                string xml = "<MyXml><InnerXml location=\"inside\"/></MyXml>";
                Console.WriteLine("Wysyanie komunikatu do kolejki komunikatw: " + xml);
                mqw.SendMessage("Etykieta komunikatu",xml);
                string retXml = mqw.ReadMessage();
                Console.WriteLine("Odczytanie komunikatu z kolejki komunikatw: " + retXml);
            }
            catch (Exception ex)
            {
                Debug.WriteLine(ex.ToString());
            }
        }

        #region MQWorker class
        class MQWorker
        {
            private string _mqPathName;
            MessageQueue _queue = null;

            public MQWorker(string queuePathName)
            {
                if (string.IsNullOrEmpty(queuePathName))
                    throw new ArgumentNullException("queuePathName");

                _mqPathName = queuePathName;

                SetUpQueue();
            }

            private void SetUpQueue()
            {
                // sprawdzenie, czy kolejka istnieje
                // jeli nie - utworzenie jej
                if (!MessageQueue.Exists(_mqPathName))
                {
                    try
                    {
                        _queue = MessageQueue.Create(_mqPathName);
                    }
                    catch (MessageQueueException mqex)
                    {
                        // sprawdzenie, czy mamy do czynienia z komputerem zespou
                        if (mqex.MessageQueueErrorCode == MessageQueueErrorCode.UnsupportedOperation)
                        {
                            string origPath = _mqPathName;
                            // musi to by prywatna kolejka w trybie stacji roboczej
                            int index = _mqPathName.ToLower().IndexOf("private$");
                            if (index == -1)
                            {
                                // pobranie pierwszego \
                                index = _mqPathName.IndexOf(@"\");
                                // wstawienie private$\ po wpisie serwera
                                _mqPathName = _mqPathName.Insert(index + 1, @"private$\");
                                // kolejne try
                                try
                                {
                                    if (!MessageQueue.Exists(_mqPathName))
                                        _queue = MessageQueue.Create(_mqPathName);
                                    else
                                        _queue = new MessageQueue(_mqPathName);
                                }
                                catch (Exception)
                                {
                                    // ustawienie wyjtki wewntrznego
                                    throw new Exception("Nie powiodo si utworzenie kolejki komunikatw z " + origPath +
                                        " lub " + _mqPathName, mqex);
                                }
                            }
                        }
                    }
                }
                else
                {
                    _queue = new MessageQueue(_mqPathName);
                }
            }

            public void SendMessage(string label, string body)
            {
                if (_queue != null)
                {
                    Message msg = new Message();
                    // nadanie etykiety komunikatowi
                    msg.Label = label;
                    // pokrycie domylnego formatowania XML formatowaniem binarnym,
                    // ktry jest szybszy (kosztem czytelnoci w trakcie debugowania)
                    msg.Formatter = new BinaryMessageFormatter();
                    // utrzymanie komunikatu
                    // (powoduje zapisanie komunikatu na dysku)
                    msg.Recoverable = true;
                    msg.Body = body;
                    _queue.Send(msg);
                }
            }

            public string ReadMessage()
            {
                Message msg = null;
                msg = _queue.Receive();
                msg.Formatter = new BinaryMessageFormatter();
                return (string)msg.Body;
            }

        }
        #endregion

        #endregion

        #region 20.5 Odnajdywanie cieki do biecej wersji .NET Framework
        public static string GetCurrentFrameworkPath()
        {
            return System.Runtime.InteropServices.RuntimeEnvironment.GetRuntimeDirectory();
        }
        #endregion

        #region 20.6 Ustalanie wersji podzespou zarejestrowanych w globalnej pamici podrcznej podzespow
        public static void PrintGACRegisteredVersions(string assmFileName)
        {
            Console.WriteLine("Wyszukiwanie wpisw GAC dla {0}\r\n", assmFileName);
            // odczytanie nazwy pliku bez rozszerzenia, poniewa jest to nazwa podkatalogu
            // w GAC, w ktrym podzesp jest zarejestrowany
            string assmFileNameNoExt = Path.GetFileNameWithoutExtension(assmFileName);

            // trzeba wyszuka zarwno natywne, jak i "zwyke" wersje plikw dll i exe
            string searchDLL = assmFileNameNoExt + ".dll";
            string searchEXE = assmFileNameNoExt + ".exe";
            string searchNIDLL = assmFileNameNoExt + ".ni.dll";
            string searchNIEXE = assmFileNameNoExt + ".ni.exe";
            
            // odczytanie cieki do GAC przy uyciu metody GetWinDir z Receptury 20.7
            string gacPath = GetWinDir() + @"\ASSEMBLY\";

            // odczytanie wszystkich pochodnych wersji pliku w GAC
            string [] dllFiles = Directory.GetFiles(gacPath, searchDLL, SearchOption.AllDirectories);
            string [] exeFiles = Directory.GetFiles(gacPath, searchEXE, SearchOption.AllDirectories);
            string [] niDllFiles = Directory.GetFiles(gacPath, searchNIDLL, SearchOption.AllDirectories);
            string [] niExeFiles = Directory.GetFiles(gacPath, searchNIEXE, SearchOption.AllDirectories);

            ArrayList files = new ArrayList(5);
            files.AddRange(dllFiles);
            files.AddRange(exeFiles);
            files.AddRange(niDllFiles);
            files.AddRange(niExeFiles);

            foreach (string file in files)
            {
                // odczytanie i wywietlenie informacji o wersjach
                FileVersionInfo fileVersion = FileVersionInfo.GetVersionInfo(file);
                if (file.IndexOf("NativeImage") != -1)
                {
                    Console.WriteLine("Znaleziono {0} w GAC z wersj natywn znajdujc si w {1}",
                        assmFileNameNoExt, Path.GetDirectoryName(file));
                }
                else
                {
                    Console.WriteLine("Znaleziono {0} w GAC, w lokalizacji {1}; informacja o wersji:\r\n{2}",
                        assmFileNameNoExt, Path.GetDirectoryName(file), fileVersion.ToString());
                }
            }
        }
		#endregion

        #region 20.7 Odczytywanie cieki do katalogu Windows
        public static string GetWinDir()
        {
            string sysDir = Environment.GetFolderPath(Environment.SpecialFolder.System);
            return Path.GetFullPath(sysDir + @"\..");
        }
		#endregion

        #region 20.8 Przechwytywanie danych wyjciowych ze standardowego strumienia wyjcia
        public static void RedirectOutput()
        {
            try 
            {
                Console.WriteLine("Przechwycenie standardowego wyjcia!");
                using (StreamWriter writer = new StreamWriter(@"c:\log.txt"))
                {
                    // przechwycenie stdout dla wasnych celw...
                    Console.SetOut(writer);

                    Console.WriteLine("Zapisywanie na konsol... NIE!");

                    for (int i = 0; i < 10; i++)
                        Console.WriteLine(i);
                }
            }
            catch(IOException e) 
            {
                Debug.WriteLine(e.ToString());
                return ;            
            }

            // przywrcenie standardowego strumienia wyjcia,
            // by mc wywietli komunikat kocowy
            using (StreamWriter standardOutput = new StreamWriter(Console.OpenStandardOutput()))
            {
                standardOutput.AutoFlush = true;
                Console.SetOut(standardOutput);
                Console.WriteLine("Przywrcenie standardowego wyjcia!");
            }
        }
		#endregion

        #region 20.9 Uruchamianie kodu w jego wasnej domenie AppDomain
        public class RunMe : MarshalByRefObject
        {
            public RunMe()
            {
                PrintCurrentAppDomainName();
            }

            public void PrintCurrentAppDomainName()
            {
                string name = AppDomain.CurrentDomain.FriendlyName;
                Console.WriteLine("Pozdrowienia z {0}!", name);
            }
        }
        public static void RunCodeInNewAppDomain()
        {
            AppDomain myOwnAppDomain = AppDomain.CreateDomain("MojaAppDomain");
            // wywietlenie nazwy biecej domeny AppDomain
            RunMe rm = new RunMe();
            rm.PrintCurrentAppDomainName();

            // utworzenie egzemplarza klasy RunMe w nowej domenie AppDomain
            Type adType = typeof(RunMe);
            ObjectHandle objHdl = myOwnAppDomain.CreateInstance(adType.Module.Assembly.FullName, adType.FullName);
            // odsonicie odwoania
            RunMe adRunMe = (RunMe)objHdl.Unwrap();

            // wywoanie na przyborniku
            adRunMe.PrintCurrentAppDomainName();

            // wyadowanie domeny AppDomain
            AppDomain.Unload(myOwnAppDomain);
        }

		#endregion

        #region 20.10 Ustalanie wersji systemu operacyjnego oraz pakietu Service Pack
        public static string GetOSAndServicePack()
        {
            // odczytanie informacji o biecym systemie operacyjnym
            OperatingSystem os = Environment.OSVersion;
            string osText = "";
            // jeli wersja ma nr 5, jest to Win2K, XP, lub 2003
            if (os.Version.Major == 5)
            {
                switch (os.Version.Minor)
                {
                    case 0: osText = "Windows 2000";
                        break;
                    case 1: osText = "Windows XP";
                        break;
                    case 2: osText = "Windows Server 2003";
                        break;
                    // dziaanie domylne, zwykle zwracajce "Microsoft Windows NT",
                    // poniewa korzysta z PlatformID
                    default: osText = os.ToString();
                        break;
                }
            }
            else
            {
                // zapewne NT4, bo .NET nie dziaa w Win9X...
                osText = os.VersionString;
            }

            // odczytanie informacji o pakiecie Service Pack
            string spText = os.ServicePack;
            // skonstruowanie wynikowego cigu znakw
            return string.Format("{0} {1}", osText, spText);
        }
		#endregion

    }
}
